/**
 * \file: sdc_clt.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: SDC Command Line Tool
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>


#include <sdc_keystore_keys.h>
#include <sdc_op_conv.h>
#include <sdc_op_adv.h>
#include <sdc_random.h>
#include <private/sdc_helper.h>

#include "sdc_ctl.h"
#include "sdc_ctl_action.h"
#include "sdc_ctl_args.h"


#define SDC_CTL_KEYLEN_DEFAULT 32

#define INFO_SEP "-------------------------------------------------------------\n"
#define KEYLEN_OP_INFO_MAX 30
#define MECHANISM_NAME_MAX 30

#define KEYDESC_LEN 30
#define KEYDESC_UNSPEC_KEY "(UNSPEC)"
#define KEYDESC_KEYSTORE_FMT "(KEYSTORE %u)"
#define KEYDESC_BUILTIN_FMT "(BUILTIN %zu/%u/%u)"

typedef struct {
    char key_desc[KEYDESC_LEN];
    action_key_spec_t key_spec;
} sdc_ctl_key_t;

typedef enum formatted_autoload_op {
    AUTOLOAD_FORMATTED_UNWRAP,
    AUTOLOAD_FORMATTED_DECRYPT,
    AUTOLOAD_FORMATTED_VERIFY
} formatted_autoload_op_t;

/* prototypes */
static sdc_error_t read_file_limit (const char *name, sdc_unittest_data_t *data, size_t limit, bool limit_length, logging_t log_level);
static sdc_error_t read_file (const char *name, sdc_unittest_data_t *data, logging_t log_level);
static sdc_error_t write_file (const char *name, const sdc_unittest_data_t *data, logging_t log_level);

static void key_invalidate(sdc_ctl_key_t *key);
static void key_set_keystore(sdc_ctl_key_t *key, sdc_key_id_t kid);
static void key_set_builtin(sdc_ctl_key_t *key, const action_key_spec_t *action_key_spec);
static sdc_error_t check_key_arguments_common(const action_key_spec_t *action_key_spec, sdc_ctl_key_t *key, bool *read_from_format);
static sdc_error_t load_session_key_common(sdc_session_t *session, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t formatted_autoload_key(sdc_session_t *session,
                                          sdc_ctl_key_t *key, formatted_autoload_op_t autoload_op,
                                          const sdc_unittest_data_t *input_data,
                                          logging_t log_level);

static sdc_error_t perform_install_action (const action_t *action, sdc_ctl_key_t *last_key, logging_t log_level);
static sdc_error_t perform_remove_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_non_formatted_wrap_unwrap_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_formatted_wrap_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_formatted_unwrap_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_non_formatted_encrypt_decrypt_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_formatted_encrypt_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_formatted_decrypt_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_formatted_sign_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_formatted_verify_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t dump_operation_info (char *op_text, const sdc_mechanism_kinds_t kind, logging_t log_level);
static sdc_error_t perform_non_formatted_sign_verify_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level);
static sdc_error_t perform_non_formatted_dgst_action (const action_t *action, logging_t log_level);
static sdc_error_t perform_rng_ops (const action_t *action, logging_t log_level);
static sdc_error_t perform_arch_info_action (const action_t *action, logging_t log_level);

static sdc_error_t read_file_limit (const char *name, sdc_unittest_data_t *data, size_t limit, bool limit_length, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    FILE *fp;
    size_t readbytes;
    long int filelen;
    long int longlimit;

    data->data = NULL;
    longlimit = limit;

    fp = fopen(name, "rb");
    if (fp == NULL) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Couldn't open file %s.\n", name);
        return SDC_UNKNOWN_ERROR;
    }

    /* Get size of data */
    fseek(fp, 0L, SEEK_END);
    filelen = ftell(fp);
    fseek(fp, 0L, SEEK_SET);

    if (filelen < 0) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Invalid length of file %s.\n", name);
        err = SDC_UNKNOWN_ERROR;
    }

    if (limit_length) {
        if (filelen < longlimit) {
            if (log_level > LOGGING_QUIET)
                fprintf(stderr, "Invalid length of file %s - at least %ld expected - is %ld.\n", name, longlimit, filelen);
            err = SDC_UNKNOWN_ERROR;
        } else {
            filelen = longlimit;
        }
    }

    if (err == SDC_OK) {
        data->len = (size_t)filelen;
        data->data = malloc(data->len);
        if ((data->data == NULL) && (data->len > 0)) {
            if (log_level > LOGGING_QUIET)
                fprintf(stderr, "Couldn't allocate memory to read %s.\n", name);
            err = SDC_NO_MEM;
        }
    }

    if ((err == SDC_OK) && (data->len > 0)) {
        readbytes = fread(data->data, 1, data->len, fp);
        if (data->len != readbytes) {
            if (log_level > LOGGING_QUIET)
                fprintf(stderr, "Unexpected amount of data read (%zu) from file %s.\n", readbytes, name);
            err = SDC_UNKNOWN_ERROR;
        }
    }

    if (err != SDC_OK) {
        if (data->data != NULL) {
            free(data->data);
            data->data = NULL;
        }
    }

    fclose (fp);

    return err;
}

static sdc_error_t read_file (const char *name, sdc_unittest_data_t *data, logging_t log_level)
{
    return read_file_limit(name, data, 0, false, log_level);
}

static sdc_error_t write_file (const char *name, const sdc_unittest_data_t *data, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    FILE *fp;
    size_t writebytes;

    fp = fopen(name, "wb");
    if (fp == NULL) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Couldn't open file %s.\n", name);
        return SDC_UNKNOWN_ERROR;
    }

    writebytes = fwrite(data->data, 1, data->len, fp);
    if (data->len != writebytes) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Unexpected amount of data written (%zu) to file %s.\n", writebytes, name);
        err = SDC_UNKNOWN_ERROR;
    }

    fclose (fp);

    return err;
}


/* common function for init/update/finalize processing */
static sdc_error_t allocate_buffers(sdc_unittest_data_t *in_buf, size_t max_in_len,
                                    sdc_unittest_data_t *out_buf, size_t max_out_len,
                                    logging_t log_level)
{
    in_buf->len = max_in_len;
    out_buf->len = max_out_len;

    in_buf->data = malloc(in_buf->len);
    out_buf->data = malloc(out_buf->len);

    if ((in_buf->data == NULL) || (out_buf->data == NULL)) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Failed to allocate buffers (%zu/%zu).\n",
                    in_buf->len, out_buf->len);

        free(in_buf->data);
        in_buf->data = NULL;

        free(out_buf->data);
        out_buf->data = NULL;

        return SDC_NO_MEM;
    }

    return SDC_OK;
}

/* common function for init/update/finalize processing */
static sdc_error_t allocate_input_buffers(sdc_unittest_data_t *in_buf, size_t max_in_len, logging_t log_level)
{
    in_buf->len = max_in_len;

    in_buf->data = malloc(in_buf->len);
    if (in_buf->data == NULL) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Failed to allocate input buffers (%zu).\n", in_buf->len);

        return SDC_NO_MEM;
    }

    return SDC_OK;
}

static sdc_error_t open_input(FILE **in, const char *infile, size_t *infile_len, logging_t log_level)
{
    long filelen;

    *in = fopen(infile, "r");
    if (*in == NULL) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Failed to open input for reading (%s).\n", infile);
        return SDC_UNKNOWN_ERROR;
    }

    /* Get size of data */
    fseek(*in, 0L, SEEK_END);
    filelen = ftell(*in);
    fseek(*in, 0L, SEEK_SET);

    if (filelen < 0) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Invalid input file %s lenght.\n", infile);
        fclose(*in);
        *in = NULL;
        return SDC_UNKNOWN_ERROR;
    } else {
        *infile_len = (size_t) filelen;
    }

    if (log_level >= LOGGING_VERBOSE) {
        printf("Input file length %zu\n", *infile_len);
    }

    return SDC_OK;
}

static sdc_error_t open_input_output(FILE **in, const char *infile, size_t *infile_len,
                                     FILE **out, const char *outfile,
                                     logging_t log_level)
{
    sdc_error_t err;

    err = open_input(in, infile, infile_len, log_level);
    if(err != SDC_OK)
        return err;

    *out = fopen(outfile, "w");
    if (*out == NULL) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Failed to open output for writing (%s).\n", outfile);

        fclose(*in);
        *in = NULL;
        return SDC_UNKNOWN_ERROR;
    }

    return SDC_OK;
}

static sdc_error_t read_buffer(FILE *in, sdc_unittest_data_t *in_buf,
                               size_t *read_len, logging_t log_level)
{
    long filePos;
    size_t max_in;

    max_in = in_buf->len;

    *read_len = fread(in_buf->data, 1, max_in, in);
    if (*read_len == 0) {
        if (ferror(in) != 0) {
            if (log_level > LOGGING_QUIET)
                fprintf(stderr, "Error reading file.\n");
            return SDC_UNKNOWN_ERROR;
        }
    }

    if (log_level >= LOGGING_VERBOSE) {
        /* Get size of data */
        filePos = ftell(in);

        printf("Input file position %lu\n", filePos);
    }

    return SDC_OK;
}

static sdc_error_t write_buffer(FILE *out, sdc_unittest_data_t *out_buf,
                                size_t len, logging_t log_level)
{
    long filePos;
    size_t written_bytes;

    if (len > out_buf->len) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Invalid write request. Can't write %zu from a %zu bytes buffer.\n", len, out_buf->len);
        return SDC_UNKNOWN_ERROR;
    }

    written_bytes = fwrite(out_buf->data, 1, len, out);
    if (len != written_bytes) {
        if (log_level > LOGGING_QUIET)
            fprintf(stderr, "Unexpected amount of data written (%zu - expected %zu).\n", written_bytes, len);
        return SDC_UNKNOWN_ERROR;
    }

    if (log_level >= LOGGING_VERBOSE) {
        /* Get size of data */
        filePos = ftell(out);

        printf("Output file position %lu\n", filePos);
    }

    return SDC_OK;
}



static void key_invalidate(sdc_ctl_key_t *key)
{
    memset(key, 0, sizeof(sdc_ctl_key_t));
    snprintf(key->key_desc, KEYDESC_LEN, KEYDESC_UNSPEC_KEY);
    key->key_spec.source = ACTION_KEY_UNSPECIFIED;
}

static void key_set_keystore(sdc_ctl_key_t *key, sdc_key_id_t kid)
{
    memset(key, 0, sizeof(sdc_ctl_key_t));
    snprintf(key->key_desc, KEYDESC_LEN, KEYDESC_KEYSTORE_FMT, kid);
    key->key_spec.source = ACTION_KID_ARG;
    key->key_spec.keystore.kid = kid;
}

static void key_set_builtin(sdc_ctl_key_t *key, const action_key_spec_t *action_key_spec)
{
    size_t key_bits=0;

    memset(key, 0, sizeof(sdc_ctl_key_t));
    key->key_spec.source = ACTION_BUILTIN_ARG;
    key->key_spec.builtin.len = action_key_spec->builtin.len;
    key->key_spec.builtin.permissions = action_key_spec->builtin.permissions;
    key->key_spec.builtin.modifier_filename = action_key_spec->builtin.modifier_filename;
    key->key_spec.builtin.secret_modifier = action_key_spec->builtin.secret_modifier;

    if (action_key_spec->builtin.uid_set)
        key->key_spec.builtin.uid = action_key_spec->builtin.uid;
    else
        key->key_spec.builtin.uid = geteuid();

    if (action_key_spec->builtin.gid_set)
        key->key_spec.builtin.gid = action_key_spec->builtin.gid;
    else
        key->key_spec.builtin.gid = getegid();

    sdc_key_len_get_bits(key->key_spec.builtin.len, &key_bits);

    snprintf(key->key_desc, KEYDESC_LEN,
             KEYDESC_BUILTIN_FMT,
             key_bits,
             key->key_spec.builtin.uid,
             key->key_spec.builtin.gid);
}


/* common function (included at each operation) to check key specification in the arguments and update key */
static sdc_error_t check_key_arguments_common(const action_key_spec_t *action_key_spec, sdc_ctl_key_t *key, bool *read_from_format)
{
    sdc_error_t err = SDC_OK;

    if (read_from_format)
        *read_from_format = false;

    switch (action_key_spec->source) {
    case ACTION_KID_ARG:
        key_set_keystore(key, action_key_spec->keystore.kid);
        break;
    case ACTION_KEY_LAST:
        /* reuse last key */
        break;
    case ACTION_BUILTIN_ARG:
        key_set_builtin(key, action_key_spec);
        break;
    case ACTION_KEY_UNSPECIFIED:
        /* copy modifier filename */
        key->key_spec.builtin.modifier_filename = action_key_spec->builtin.modifier_filename;

        /* unspecified key is only allowed in combination with formatted operations */
        if (read_from_format) {
            *read_from_format = true;
            break;
        }
    /* FALLTHROUGH */
    default:
        fprintf(stderr, "Invalid key type\n");
        err = SDC_UNKNOWN_ERROR;
    }

    return err;
}

static sdc_error_t load_session_key_common(sdc_session_t *session, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_permissions_t *builtin_perm = NULL;
    sdc_unittest_data_t builtin_mod;
    bool builtin_mod_secret = false;

    builtin_mod.data = NULL;
    builtin_mod.len = 0;

    switch (key->key_spec.source) {
    case ACTION_KID_ARG:
        err = sdc_session_load_storage_key(session, key->key_spec.keystore.kid);
        break;
    case ACTION_BUILTIN_ARG:
        err = sdc_permissions_alloc(&builtin_perm);

        if (err == SDC_OK)
            err = sdc_set_default_permissions_and_gid (builtin_perm, key->key_spec.builtin.gid);

        if (err == SDC_OK)
            err = sdc_permissions_set_uid(builtin_perm, key->key_spec.builtin.uid);

        if ((err == SDC_OK) && (key->key_spec.builtin.permissions))
            err = sdc_permissions_apply_string(builtin_perm,
                                               key->key_spec.builtin.permissions,
                                               strlen(key->key_spec.builtin.permissions));

        if ((err == SDC_OK) && (key->key_spec.builtin.modifier_filename)) {
            builtin_mod_secret = key->key_spec.builtin.secret_modifier;
            err = read_file(key->key_spec.builtin.modifier_filename, &builtin_mod, log_level);
        }

        if (err == SDC_OK)
            err = sdc_session_load_builtin_key(session,
                                               SDC_KEY_FMT_SIMPLE_SYM,
                                               key->key_spec.builtin.len,
                                               builtin_mod.data, builtin_mod.len, builtin_mod_secret,
                                               builtin_perm);

        break;
    case ACTION_KEY_UNSPECIFIED:
    case ACTION_KEY_LAST:
    /* FALLTHROUGH */
    default:
        fprintf(stderr, "Invalid key type\n");
        err = SDC_UNKNOWN_ERROR;
    }

    if (builtin_perm) {
        tmp_err = sdc_permissions_free(builtin_perm);
        if (err == SDC_OK)
            err = tmp_err;
    }

    free (builtin_mod.data);

    return err;
}

static sdc_error_t formatted_autoload_key(sdc_session_t *session,
                                          sdc_ctl_key_t *key, formatted_autoload_op_t autoload_op,
                                          const sdc_unittest_data_t *input_data,
                                          logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_unittest_data_t builtin_mod;

    builtin_mod.data = NULL;
    builtin_mod.len = 0;

    /* this should not happen */
    if (key->key_spec.source != ACTION_KEY_UNSPECIFIED)
        return SDC_UNKNOWN_ERROR;

    if (key->key_spec.builtin.modifier_filename) {
        err = read_file(key->key_spec.builtin.modifier_filename, &builtin_mod, log_level);

        if (err == SDC_OK) {
            switch (autoload_op) {
            case AUTOLOAD_FORMATTED_UNWRAP:
                err = sdc_unwrap_formatted_autoload_key_with_secret_mod(session,
                                                                        input_data->data, input_data->len,
                                                                        builtin_mod.data, builtin_mod.len);
                break;
            case AUTOLOAD_FORMATTED_DECRYPT:
                err = sdc_decrypt_formatted_autoload_key_with_secret_mod(session,
                                                                         input_data->data, input_data->len,
                                                                         builtin_mod.data, builtin_mod.len);
                break;
            case AUTOLOAD_FORMATTED_VERIFY:
                err = sdc_verify_formatted_autoload_key_with_secret_mod(session,
                                                                        input_data->data, input_data->len,
                                                                        builtin_mod.data, builtin_mod.len);
                break;
            default:
                err = SDC_UNKNOWN_ERROR;
            }
        }

        free (builtin_mod.data);
    } else {
        switch (autoload_op) {
        case AUTOLOAD_FORMATTED_UNWRAP:
            err = sdc_unwrap_formatted_autoload_key(session, input_data->data, input_data->len);
            break;
        case AUTOLOAD_FORMATTED_DECRYPT:
            err = sdc_decrypt_formatted_autoload_key(session, input_data->data, input_data->len);
            break;
        case AUTOLOAD_FORMATTED_VERIFY:
            err = sdc_verify_formatted_autoload_key(session, input_data->data, input_data->len);
            break;
        default:
            err = SDC_UNKNOWN_ERROR;
        }
    }

    return err;
}


static sdc_error_t perform_install_plain (
    const char *input_filename,
    sdc_key_len_t len,
    sdc_key_fmt_t fmt,
    sdc_keysecret_enc_t key_data_enc,
    sdc_key_id_t *kid,
    sdc_key_id_options_t kid_opt,
    sdc_key_storage_options_t stor_opt,
    const sdc_permissions_t *permissions,
    logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    size_t read_limit = 0;
    bool has_read_limit = false;
    sdc_unittest_data_t key_data;

    key_data.data = NULL;

    /* limit length of read file if length is specified
     * format is simple symmetric */
    if ((len != SDC_KEY_LEN_UNKNOWN) && (fmt == SDC_KEY_FMT_SIMPLE_SYM)) {
        has_read_limit = true;
        err = sdc_key_len_get_bytes(len, &read_limit);
    }

    if (err == SDC_OK)
        err = read_file_limit(input_filename, &key_data, read_limit, has_read_limit, log_level);

    /* use file length if no length is specified */
    if ((len == SDC_KEY_LEN_UNKNOWN) && (err == SDC_OK)) {
        err = sdc_key_len_from_bytes(key_data.len, &len);
    }

    if (err == SDC_OK) {
        if (log_level >= LOGGING_VERBOSE) {
            printf("Insert plain %s key (len %zu/%s) into key storage (",
                   stor_opt == SDC_VOLATILE_STORAGE_KEY ? "volatile" : "persistent",
                   key_data.len,sdc_key_fmt_get_name(fmt));
            if (kid_opt == SDC_CREATE_KEY_AUTOMATIC_ID) {
                printf("KID is determined automatically)\n");
            } else {
                printf("fixed KID %u)\n", *kid);
            }
        }

        err = sdc_insert_storage_key(
            fmt, len,
            kid, kid_opt, stor_opt,
            permissions,
            key_data_enc, key_data.data, key_data.len);
    }

    if (key_data.data)
        free (key_data.data);

    return err;
}

static sdc_error_t perform_install_random (
    sdc_key_len_t len,
    sdc_key_fmt_t fmt,
    sdc_key_id_t *kid,
    sdc_key_id_options_t kid_opt,
    sdc_key_storage_options_t stor_opt,
    const sdc_permissions_t *permissions,
    logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    size_t rand_len = 0;

    if (len == SDC_KEY_LEN_UNKNOWN)
        err = sdc_key_len_from_bytes(SDC_CTL_KEYLEN_DEFAULT, &len);

    if (fmt != SDC_KEY_FMT_SIMPLE_SYM)
        err = SDC_NOT_SUPPORTED;

    if (err == SDC_OK) {
        err = sdc_key_len_get_bytes(len, &rand_len);
    }

    if (err == SDC_OK) {
        if (log_level >= LOGGING_VERBOSE) {
            printf("Insert random %s key (len %zu/%s) into key storage (",
                   stor_opt == SDC_VOLATILE_STORAGE_KEY ? "volatile" : "persistent",
                   rand_len, sdc_key_fmt_get_name(fmt));
            if (kid_opt == SDC_CREATE_KEY_AUTOMATIC_ID) {
                printf("KID is determined automatically)\n");
            } else {
                printf("fixed KID %u)\n", *kid);
            }
        }
        err = sdc_generate_random_storage_key(
            kid, kid_opt,
            rand_len,
            stor_opt, permissions);
    }

    return err;
}

static sdc_error_t perform_install_wrapped_key (
    const char *input_filename,
    sdc_key_len_t len,
    sdc_key_fmt_t fmt,
    sdc_keysecret_enc_t key_data_enc,
    sdc_key_id_t *kid,
    sdc_key_id_options_t kid_opt,
    sdc_key_storage_options_t stor_opt,
    const sdc_permissions_t *permissions,
    logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_unittest_data_t fkey_data;

    fkey_data.data = NULL;

    err = read_file(input_filename, &fkey_data, log_level);

    if (err == SDC_OK)
        err = sdc_open_session(&session);

    if (err == SDC_OK) {
        err = sdc_import_formatted_autoload_key_with_secret_mod(session,
                                                                fkey_data.data, fkey_data.len,
                                                                NULL, 0);
    }

    if (err == SDC_OK) {
        err = sdc_import_formatted_storage_key(session,
                                               fmt, len, key_data_enc,
                                               kid, kid_opt, stor_opt,
                                               permissions,
                                               fkey_data.data, fkey_data.len);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    if (fkey_data.data)
        free (fkey_data.data);

    return err;
}

static sdc_error_t perform_install_action (const action_t *action, sdc_ctl_key_t *last_key, logging_t log_level)
{
    const action_option_install_t *options;
    sdc_error_t err = SDC_OK;
    sdc_permissions_t *permissions = NULL;
    sdc_key_id_options_t kid_opt = SDC_CREATE_KEY_AUTOMATIC_ID;
    FILE *fp;
    sdc_key_id_t mykid = SDC_FLAG_INVALID_KID;
    sdc_key_len_t len;
    sdc_key_fmt_t fmt;
    sdc_keysecret_enc_t key_data_enc;

    options = &action->options.install_options;

    err = sdc_permissions_alloc(&permissions);

    if (err == SDC_OK) {
        if (options->gid != (gid_t)-1) {
            err = sdc_set_default_permissions_and_gid (permissions, options->gid);
        } else {
            err = sdc_set_default_permissions_current_gid (permissions);
        }
    }

    if ((err == SDC_OK) && (options->permissions))
        err = sdc_permissions_apply_string(permissions,
                                           options->permissions,
                                           strlen(options->permissions));

    if ((err != SDC_OK) && (log_level > LOGGING_QUIET)) {
        fprintf(stderr, "Key install : Error getting permissions struct\n");
    }

    /* invalidate last key */
    key_invalidate(last_key);

    if (err == SDC_OK) {
        switch (action->key_spec.source) {
        case ACTION_KID_ARG:
            mykid = action->key_spec.keystore.kid;
            kid_opt = SDC_CREATE_KEY_FIXED_ID;
            break;
        case ACTION_KEY_UNSPECIFIED:
            kid_opt = SDC_CREATE_KEY_AUTOMATIC_ID;
            break;
        case ACTION_KEY_LAST:
        case ACTION_BUILTIN_ARG:
        default:
            fprintf(stderr, "Key install : Invalid key type\n");
            err = SDC_UNKNOWN_ERROR;
        }
    }

    len = options->len;
    fmt = options->fmt;
    key_data_enc = options->enc;

    /* assign default */
    if (fmt == SDC_KEY_FMT_UNKNOWN) {
        fmt = SDC_KEY_FMT_SIMPLE_SYM;
        key_data_enc = SDC_KEY_ENC_PLAIN;
    }

    if (err == SDC_OK) {
        switch (options->type) {
        case ACTION_INSTALL_PLAIN_KEY:
            err = perform_install_plain (
                action->input,
                len, fmt, key_data_enc,
                &mykid, kid_opt, options->stor_opt,
                permissions, log_level);
            break;
        case ACTION_INSTALL_RANDOM_KEY:
            err = perform_install_random (
                len, fmt,
                &mykid, kid_opt, options->stor_opt,
                permissions, log_level);
            break;
        case ACTION_INSTALL_WRAPPED_KEY:
            err = perform_install_wrapped_key(
                action->input,
                len, fmt, key_data_enc,
                &mykid, kid_opt, options->stor_opt,
                permissions, log_level);
            break;
        default:
            err = SDC_UNKNOWN_ERROR;
        }

        if (err == SDC_OK) {
            key_set_keystore(last_key, mykid);
        }

        if ((err == SDC_OK) && (options->kid_file != NULL)) {
            err = SDC_UNKNOWN_ERROR;
            fp = fopen(options->kid_file, "w");
            if (fp != NULL) {
                if (0 < fprintf(fp, "%u", mykid))
                    err = SDC_OK;
                fclose (fp);
            }
        }

        if (err == SDC_OK) {
            if (((kid_opt == SDC_CREATE_KEY_AUTOMATIC_ID) && (options->kid_file == NULL) && (log_level > LOGGING_QUIET)) ||
                (log_level >= LOGGING_VERBOSE)) {
                printf ("KID of new key is %u\n", mykid);
            }
        } else {
            if (log_level > LOGGING_QUIET) {
                fprintf(stderr, "Error installing key\n");
            }
        }
    }

    if (permissions)
        sdc_permissions_free(permissions);

    return err;
}

static sdc_error_t perform_remove_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_key_id_t mykid = SDC_FLAG_INVALID_KID;

    switch (action->key_spec.source) {
    case ACTION_KID_ARG:
        mykid = action->key_spec.keystore.kid;
        break;
    case ACTION_KEY_LAST:
        if (key->key_spec.source == ACTION_KID_ARG) {
            mykid = key->key_spec.keystore.kid;

            break;
        } /* generate error if last key is no keystore key */
    /* FALLTHROUGH */
    case ACTION_KEY_UNSPECIFIED:
    case ACTION_BUILTIN_ARG:
    default:
        fprintf(stderr, "Key remove : Invalid key type\n");
        err = SDC_UNKNOWN_ERROR;
    }

    if (err == SDC_OK) {
        key_invalidate(key);
    }

    if (log_level >= LOGGING_VERBOSE) {
        printf("Remove key with KID %u\n", mykid);
    }

    err = sdc_remove_storage_key (mykid);

    if (err != SDC_OK) {
        if (log_level > LOGGING_QUIET) {
            fprintf(stderr, "Error removing key\n");
        }
    }

    return err;
}

static sdc_error_t sign_unwrap_get_tag_info(const action_t *action, sdc_unittest_data_t *tag_data, logging_t log_level)
{

    sdc_error_t err = SDC_OK;
    if (action->operation == ACTION_NON_FORMATTED_UNWRAP) {
        err = read_file_limit(action->dgst_tag_spec.dgst_tag_filename,
                              tag_data, action->dgst_tag_spec.dgst_tag_len,
                              action->dgst_tag_spec.dgst_tag_len_specified,
                              log_level);
    } else {
        if (action->dgst_tag_spec.dgst_tag_len_specified) {
            tag_data->len = action->dgst_tag_spec.dgst_tag_len;
        } else {
            tag_data->len = SDC_TAG_USE_DEFAULT;
        }
    }

    return err;
}

static sdc_error_t wrap_unwrap_one_shot (const action_t *action,
                                         sdc_session_t *session,
                                         const sdc_wrap_unwrap_type_t *wrap_unwrap_type,
                                         sdc_unittest_data_t *iv,
                                         sdc_unittest_data_t *tag_data,
                                         logging_t log_level) {
    sdc_error_t err = SDC_OK;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t output_data;

    input_data.data = NULL;
    input_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = read_file(action->input, &input_data, log_level);

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_UNWRAP) {
            err = sdc_unwrap(session, wrap_unwrap_type, input_data.data,
                             input_data.len, NULL, 0, iv->data, iv->len,
                             tag_data->data, tag_data->len, &output_data.data,
                             &output_data.len);
        } else {
            err = sdc_wrap(session, wrap_unwrap_type, input_data.data,
                           input_data.len, NULL, 0, &iv->data, &iv->len,
                           &output_data.data, &output_data.len, &tag_data->data,
                           &tag_data->len);
        }
    }

    /* store final output data */
    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    if (input_data.data)
        free (input_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t wrap_unwrap_init_update_final (const action_t *action,
                                                  sdc_session_t *session,
                                                  const sdc_wrap_unwrap_type_t *wrap_unwrap_type,
                                                  sdc_unittest_data_t *iv,
                                                  sdc_unittest_data_t *tag_data,
                                                  logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_wrap_unwrap_desc_t *desc = NULL;
    sdc_unittest_data_t in_buf;
    sdc_unittest_data_t out_buf;
    size_t infile_len = 0;
    size_t in_len;
    size_t out_len;
    size_t max_in_len = 0;
    size_t max_out_len = 0;
    FILE *in = NULL;
    FILE *out = NULL;

    in_buf.data = NULL;
    in_buf.len = 0;
    out_buf.data = NULL;
    out_buf.len = 0;

    err = sdc_wrap_unwrap_desc_alloc(&desc);

    if (err == SDC_OK)
        err = sdc_wrap_unwrap_desc_fill(session, wrap_unwrap_type, desc, max_in_len);

    if (err == SDC_OK) {
        if (action->huge_data_opts.chunk_length_set)
            max_in_len = action->huge_data_opts.chunk_length;
        else
            err = sdc_wrap_unwrap_desc_get_max_chunk_len (desc, &(max_in_len));
    }

    if (err == SDC_OK)
        err = open_input_output(&in, action->input, &infile_len, &out, action->output, log_level);

    /* it is useless to allocate more input buffer than the length of the file - so use this as a limit */
    if (max_in_len > infile_len)
        max_in_len = infile_len;

    if (err == SDC_OK)
        err = sdc_wrap_unwrap_desc_fill(session, wrap_unwrap_type, desc, max_in_len);

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_UNWRAP)
            err = sdc_unwrap_get_update_len(session,
                                            wrap_unwrap_type,
                                            desc,
                                            max_in_len,
                                            &max_out_len);
        else
            err = sdc_wrap_get_update_len(session,
                                          wrap_unwrap_type,
                                          desc,
                                          max_in_len,
                                          &max_out_len);
        if ((err != SDC_OK) && (log_level > LOGGING_QUIET))
            fprintf(stderr, "Failed to determine output buffer length.\n");
    }

    if (err == SDC_OK)
        err = allocate_buffers(&in_buf, max_in_len,
                               &out_buf, max_out_len,
                               log_level);

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_UNWRAP)
            err = sdc_unwrap_init(session,
                                  wrap_unwrap_type,
                                  desc,
                                  infile_len,
                                  0,
                                  tag_data->len,
                                  iv->data, iv->len);
        else
            err = sdc_wrap_init(session,
                                wrap_unwrap_type,
                                desc,
                                infile_len,
                                0,
                                &tag_data->len,
                                &iv->data, &iv->len);
    }

    while (err == SDC_OK) {
        /* fill next input buffer */
        err = read_buffer(in, &in_buf, &in_len, log_level);

        if ((err != SDC_OK) || (in_len == 0))
            break; /* either no more data or error */

        /* set avail len */
        out_len = out_buf.len;

        /* process */
        if (action->operation == ACTION_NON_FORMATTED_UNWRAP)
            err = sdc_unwrap_update(session,
                                    wrap_unwrap_type,
                                    desc,
                                    in_buf.data, in_len,
                                    out_buf.data, &out_len);
        else
            err = sdc_wrap_update(session,
                                  wrap_unwrap_type,
                                  desc,
                                  in_buf.data, in_len,
                                  out_buf.data, &out_len);

        /* store output buffer */
        if ((err == SDC_OK) && (out_len > 0))
            err = write_buffer(out, &out_buf, out_len, log_level);
    }

    if (err == SDC_OK) {
        /* set avail len */
        out_len = 16;

        /* get final output data */
        if (action->operation == ACTION_NON_FORMATTED_UNWRAP)
            err = sdc_unwrap_finalize(session,
                                      wrap_unwrap_type,
                                      desc,
                                      out_buf.data, &out_len,
                                      tag_data->data, tag_data->len);
        else
            err = sdc_wrap_finalize(session,
                                    wrap_unwrap_type,
                                    desc,
                                    out_buf.data, &out_len,
                                    &tag_data->data, tag_data->len);

        /* store final output data */
        if ((err == SDC_OK) && (out_len > 0))
            err = write_buffer(out, &out_buf, out_len, log_level);
    }

    if (in)
        fclose(in);
    if (out)
        fclose(out);
    if (in_buf.data)
        free (in_buf.data);
    if (out_buf.data)
        free (out_buf.data);
    if (desc) {
        tmp_err = sdc_wrap_unwrap_desc_free(desc);
        if (err == SDC_OK)
            err = tmp_err;
    }

    return err;
}

static sdc_error_t perform_non_formatted_wrap_unwrap_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_wrap_unwrap_type_t *allocated_wrap_unwrap_type = NULL;
    const sdc_wrap_unwrap_type_t *wrap_unwrap_type = NULL;
    sdc_unittest_data_t tag_data;
    sdc_unittest_data_t iv_data;
    action_iv_mode_t iv_mode = ACTION_IV_DEFAULT;

    iv_data.data = NULL;
    iv_data.len = 0;
    tag_data.data = NULL;
    tag_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, NULL);

    iv_mode = action->iv_spec.iv_mode;
    if (iv_mode == ACTION_IV_DEFAULT) {
        iv_mode = ACTION_IV_NOIV;
    }

    if (err == SDC_OK) {
        if (log_level >= LOGGING_VERBOSE) {
            if (action->operation == ACTION_NON_FORMATTED_WRAP) {
                if (iv_mode == ACTION_IV_AUTO_GEN_WRITE_FILE) {
                    printf("Wrap file %s using key %s, automatically generated iv and write result to %s, iv to %s and tag to %s\n",
                           action->input, key->key_desc,
                           action->output, action->iv_spec.iv_filename, action->dgst_tag_spec.dgst_tag_filename);
                } else {
                    if (iv_mode == ACTION_IV_FILE) {
                        printf("Wrap file %s using key %s, file IV %s and write result to %s and tag to %s\n",
                               action->input, key->key_desc, action->iv_spec.iv_filename,
                               action->output, action->dgst_tag_spec.dgst_tag_filename);
                    } else {
                        printf("Wrap file %s using key %s, without IV and write result to %s and tag to %s\n",
                               action->input, key->key_desc,
                               action->output, action->dgst_tag_spec.dgst_tag_filename);
                    }
                }
            } else {
                printf("Unwrap file %s using key %s, file IV %s, tag %s and write result to %s\n",
                       action->input, key->key_desc, action->iv_spec.iv_filename, action->dgst_tag_spec.dgst_tag_filename,
                       action->output);
            }
        }

    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_FILE)) {
        err = read_file_limit(action->iv_spec.iv_filename, &iv_data, action->iv_spec.iv_len, action->iv_spec.iv_len_specified, log_level);
    } else if (iv_mode != ACTION_IV_NOIV) {
        if (action->iv_spec.iv_len_specified) {
            iv_data.len = action->iv_spec.iv_len;
        } else {
            iv_data.len = SDC_IV_USE_DEFAULT;
        }
    }

    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        err = load_session_key_common(session, key, log_level);
    }

    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_wrap_unwrap_type_alloc(&allocated_wrap_unwrap_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_WRAP_UNWRAP, action->mech_spec.mechanism, allocated_wrap_unwrap_type);
                wrap_unwrap_type = allocated_wrap_unwrap_type;
            }

            if ((err == SDC_OK) && (action->mech_spec.mechanism_opt_set)) {
                err = sdc_wrap_unwrap_type_set_opt_bmsk(allocated_wrap_unwrap_type,
                                                        action->mech_spec.mechanism_opt);
            }
        } else {
            wrap_unwrap_type = sdc_wrap_unwrap_get_default();
        }
    }

    if (err == SDC_OK)
           err = sign_unwrap_get_tag_info(action, &tag_data, log_level);

    if (err == SDC_OK) {
        if (action->huge_data_opts.enable) {
            err = wrap_unwrap_init_update_final(action, session,
                                                wrap_unwrap_type, &iv_data, &tag_data,
                                                log_level);
        } else {
            err = wrap_unwrap_one_shot(action, session, wrap_unwrap_type,
                                       &iv_data, &tag_data, log_level);
        }
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_NOIV) && (iv_data.len > 0)) {
        fprintf(stderr, "User didn't specify IV but iv is returned\n");
        err = SDC_UNKNOWN_ERROR;
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_AUTO_GEN_WRITE_FILE)) {
        /* need to store iv */
        write_file(action->iv_spec.iv_filename, &iv_data, log_level);
    }

    if ((err == SDC_OK) && (action->operation == ACTION_NON_FORMATTED_WRAP)) {
        /* need to store tag */
        write_file(action->dgst_tag_spec.dgst_tag_filename, &tag_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_wrap_unwrap_type_free(allocated_wrap_unwrap_type);

    if (iv_data.data)
        free (iv_data.data);

    free (tag_data.data);

    return err;
}

static sdc_error_t perform_formatted_wrap_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_wrap_unwrap_type_t *allocated_wrap_unwrap_type = NULL;
    const sdc_wrap_unwrap_type_t *wrap_unwrap_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t iv_data;
    sdc_unittest_data_t output_data;
    size_t tag_len;
    action_iv_mode_t iv_mode = ACTION_IV_DEFAULT;

    input_data.data = NULL;
    input_data.len = 0;
    iv_data.data = NULL;
    iv_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, NULL);

    if (err == SDC_OK) {
        iv_mode = action->iv_spec.iv_mode;
        if (iv_mode == ACTION_IV_DEFAULT) {
            iv_mode = ACTION_IV_AUTO_GEN_NO_FILE;
        }

        if (log_level >= LOGGING_VERBOSE) {
            if (iv_mode == ACTION_IV_AUTO_GEN_NO_FILE) {
                printf("Wrap file %s using key %s, automatically generated iv and write formatted result to %s\n",
                       action->input, key->key_desc,
                       action->output);
            } else {
                if (iv_mode == ACTION_IV_FILE) {
                    printf("Wrap file %s using key %s, file IV %s and write formatted result to %s\n",
                           action->input, key->key_desc, action->iv_spec.iv_filename,
                           action->output);
                } else {
                    printf("Wrap file %s using key %s, without IV and write formatted result to %s\n",
                           action->input, key->key_desc,
                           action->output);
                }
            }
        }

        err = read_file(action->input, &input_data, log_level);
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_FILE)) {
        err = read_file_limit(action->iv_spec.iv_filename, &iv_data, action->iv_spec.iv_len, action->iv_spec.iv_len_specified, log_level);
    } else if (iv_mode != ACTION_IV_NOIV) {
        if (action->iv_spec.iv_len_specified) {
            iv_data.len = action->iv_spec.iv_len;
        } else {
            iv_data.len = SDC_IV_USE_DEFAULT;
        }
    }

    if (action->dgst_tag_spec.dgst_tag_len_specified) {
        tag_len = action->dgst_tag_spec.dgst_tag_len;
    } else {
        tag_len = SDC_TAG_USE_DEFAULT;
    }

    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        err = load_session_key_common(session, key, log_level);
    }

    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_wrap_unwrap_type_alloc(&allocated_wrap_unwrap_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_WRAP_UNWRAP, action->mech_spec.mechanism, allocated_wrap_unwrap_type);
                wrap_unwrap_type = allocated_wrap_unwrap_type;
            }
        } else {
            wrap_unwrap_type = sdc_wrap_unwrap_get_default();
        }
    }

    if (err == SDC_OK) {
        if ((iv_data.len != SDC_IV_USE_DEFAULT) || (tag_len != SDC_TAG_USE_DEFAULT)) {
            err = sdc_wrap_formatted_extended(session, wrap_unwrap_type,
                                              input_data.data, input_data.len,
                                              iv_data.data, iv_data.len,
                                              NULL, 0,
                                              tag_len,
                                              &output_data.data, &output_data.len);
        } else {
            err = sdc_wrap_formatted(session, wrap_unwrap_type,
                                     input_data.data, input_data.len,
                                     &output_data.data, &output_data.len);
        }
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_wrap_unwrap_type_free(allocated_wrap_unwrap_type);

    if (input_data.data)
        free (input_data.data);
    if (iv_data.data)
        free (iv_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t perform_formatted_unwrap_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    bool format_key;
    sdc_wrap_unwrap_type_t *wrap_unwrap_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t output_data;
    sdc_wrap_unwrap_alg_t alg;
    sdc_wrap_unwrap_blk_t blk;

    input_data.data = NULL;
    input_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, &format_key);

    if (err == SDC_OK) {
        if (log_level >= LOGGING_VERBOSE) {
            if (format_key) {
                printf("Unwrap file %s using key from format and write result to %s\n",
                       action->input, action->output);
            } else {
                printf("Unwrap file %s using key %s and write result to %s\n",
                       action->input, key->key_desc,
                       action->output);
            }
        }

        err = read_file(action->input, &input_data, log_level);
    }

    if (err == SDC_OK) {
        err = sdc_unwrap_formatted_extract_type(input_data.data, input_data.len, &wrap_unwrap_type);
    }

    if ((err == SDC_OK) && (log_level >= LOGGING_VERBOSE)) {
        err = sdc_wrap_unwrap_type_get_algorithm (wrap_unwrap_type, &alg);

        if (err == SDC_OK) {
            err = sdc_wrap_unwrap_type_get_block_mode(wrap_unwrap_type, &blk);

            if (err == SDC_OK) {
                printf("\tFormat algorithm %s, block-mode %s\n",
                       sdc_wrap_unwrap_algorithm_name(alg),
                       sdc_wrap_unwrap_block_mode_name(blk));
            }
        }
    }


    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        if (format_key) {
            err = formatted_autoload_key(session,
                                         key, AUTOLOAD_FORMATTED_UNWRAP,
                                         &input_data, log_level);
        } else {
            err = load_session_key_common(session, key, log_level);
        }
    }

    if (err == SDC_OK) {
        err = sdc_unwrap_formatted(session, wrap_unwrap_type,
                                   input_data.data, input_data.len,
                                   &output_data.data, &output_data.len);
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    if (wrap_unwrap_type) {
        tmp_err = sdc_wrap_unwrap_type_free(wrap_unwrap_type);
        if (err == SDC_OK)
            err = tmp_err;
    }

    if (input_data.data)
        free (input_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t encrypt_decrypt_one_shot (const action_t *action,
                                             sdc_session_t *session,
                                             const sdc_encrypt_decrypt_type_t *enc_dec_type,
                                             sdc_unittest_data_t *iv,
                                             logging_t log_level) {
    sdc_error_t err = SDC_OK;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t output_data;

    input_data.data = NULL;
    input_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;


    err = read_file(action->input, &input_data, log_level);

    if (action->operation == ACTION_NON_FORMATTED_DECRYPT) {
        err = sdc_decrypt(session, enc_dec_type,
                          input_data.data, input_data.len,
                          iv->data, iv->len,
                          &output_data.data, &output_data.len);
    } else {
        err = sdc_encrypt(session, enc_dec_type,
                          input_data.data, input_data.len,
                          &iv->data, &iv->len,
                          &output_data.data, &output_data.len);
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    if (input_data.data)
        free (input_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t encrypt_decrypt_init_update_final (const action_t *action,
                                                      sdc_session_t *session,
                                                      const sdc_encrypt_decrypt_type_t *enc_dec_type,
                                                      sdc_unittest_data_t *iv,
                                                      logging_t log_level) {
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_encrypt_decrypt_desc_t *desc = NULL;
    sdc_unittest_data_t in_buf;
    sdc_unittest_data_t out_buf;
    size_t infile_len = 0;
    size_t in_len;
    size_t out_len;
    FILE *in = NULL;
    FILE *out = NULL;

    size_t max_in_len = 0;
    size_t max_out_len = 0;

    in_buf.data = NULL;
    in_buf.len = 0;
    out_buf.data = NULL;
    out_buf.len = 0;

    err = sdc_encrypt_decrypt_desc_alloc(&desc);

    if (err == SDC_OK)
        err = sdc_encrypt_decrypt_desc_fill(session, enc_dec_type, desc);

    if (err == SDC_OK) {
        if (action->huge_data_opts.chunk_length_set)
            max_in_len = action->huge_data_opts.chunk_length;
        else
            err = sdc_encrypt_decrypt_desc_get_max_chunk_len (desc, &(max_in_len));
    }

    if (err == SDC_OK)
        err = open_input_output(&in, action->input, &infile_len, &out, action->output, log_level);

    /* it is useless to allocate more input buffer than the length of the file - so use this as a limit */
    if (max_in_len > infile_len)
        max_in_len = infile_len;

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_DECRYPT)
            err = sdc_decrypt_get_update_len(session,
                                                     enc_dec_type,
                                                     desc,
                                                     max_in_len,
                                                     &max_out_len);
        else
            err = sdc_encrypt_get_update_len(session,
                                                     enc_dec_type,
                                                     desc,
                                                     max_in_len,
                                                     &max_out_len);
        if ((err != SDC_OK) && (log_level > LOGGING_QUIET))
                fprintf(stderr, "Failed to determine output buffer length.\n");
    }

    if (err == SDC_OK)
        err = allocate_buffers(&in_buf, max_in_len,
                               &out_buf, max_out_len,
                               log_level);

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_DECRYPT)
            err = sdc_decrypt_init(session,
                                   enc_dec_type,
                                   desc,
                                   iv->data, iv->len);
        else
            err = sdc_encrypt_init(session,
                                   enc_dec_type,
                                   desc,
                                   &iv->data, &iv->len);
    }

    while (err == SDC_OK) {
        /* fill next input buffer */
        err = read_buffer(in, &in_buf, &in_len, log_level);

        if ((err != SDC_OK) || (in_len == 0))
            break; /* either no more data or error */

        /* set avail len */
        out_len = out_buf.len;

        /* process */
        if (action->operation == ACTION_NON_FORMATTED_DECRYPT)
            err = sdc_decrypt_update(session,
                                     enc_dec_type,
                                     desc,
                                     in_buf.data, in_len,
                                     out_buf.data, &out_len);
        else
            err = sdc_encrypt_update(session,
                                     enc_dec_type,
                                     desc,
                                     in_buf.data, in_len,
                                     out_buf.data, &out_len);

        /* store output buffer */
        if ((err == SDC_OK) && (out_len > 0))
            err = write_buffer(out, &out_buf, out_len, log_level);
    }

    if (err == SDC_OK) {
        /* set avail len */
        out_len = out_buf.len;

        /* get final output data */
        if (action->operation == ACTION_NON_FORMATTED_DECRYPT)
            err = sdc_decrypt_finalize(session,
                                       enc_dec_type,
                                       desc,
                                       out_buf.data, &out_len);
        else
            err = sdc_encrypt_finalize(session,
                                       enc_dec_type,
                                       desc,
                                       out_buf.data, &out_len);

        /* store final output data */
        if ((err == SDC_OK) && (out_len > 0))
            err = write_buffer(out, &out_buf, out_len, log_level);
    }

    if (in)
        fclose(in);
    if (out)
        fclose(out);
    if (in_buf.data)
        free (in_buf.data);
    if (out_buf.data)
        free (out_buf.data);
    if (desc) {
        tmp_err = sdc_encrypt_decrypt_desc_free(desc);
        if (err == SDC_OK)
            err = tmp_err;
    }

    return err;
}

static sdc_error_t perform_non_formatted_encrypt_decrypt_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_encrypt_decrypt_type_t *allocated_enc_dec_type = NULL;
    const sdc_encrypt_decrypt_type_t *enc_dec_type = NULL;
    sdc_unittest_data_t iv_data;
    action_iv_mode_t iv_mode = ACTION_IV_DEFAULT;

    iv_data.data = NULL;
    iv_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, NULL);

    if (err == SDC_OK) {
        iv_mode = action->iv_spec.iv_mode;
        if (iv_mode == ACTION_IV_DEFAULT) {
            iv_mode = ACTION_IV_NOIV;
        }

        if (log_level >= LOGGING_VERBOSE) {
            if (action->operation == ACTION_NON_FORMATTED_ENCRYPT) {
                if (iv_mode == ACTION_IV_AUTO_GEN_WRITE_FILE) {
                    printf("Encrypt file %s using key %s, automatically generated iv and write result to %s, iv to %s\n",
                           action->input, key->key_desc,
                           action->output, action->iv_spec.iv_filename);
                } else {
                    if (iv_mode == ACTION_IV_FILE) {
                        printf("Encrypt file %s using key %s, file IV %s and write result to %s\n",
                               action->input, key->key_desc, action->iv_spec.iv_filename,
                               action->output);
                    } else {
                        printf("Encrypt file %s using key %s, without IV and write result to %s\n",
                               action->input, key->key_desc,
                               action->output);
                    }
                }
            } else {
                printf("Decrypt file %s using key %s, file IV %s and write result to %s\n",
                       action->input, key->key_desc, action->iv_spec.iv_filename,
                       action->output);
            }
        }
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_FILE)) {
        err = read_file_limit(action->iv_spec.iv_filename, &iv_data, action->iv_spec.iv_len, action->iv_spec.iv_len_specified, log_level);
    } else if (iv_mode != ACTION_IV_NOIV) {
        if (action->iv_spec.iv_len_specified) {
            iv_data.len = action->iv_spec.iv_len;
        } else {
            iv_data.len = SDC_IV_USE_DEFAULT;
        }
    }

    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        err = load_session_key_common(session, key, log_level);
    }

    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_encrypt_decrypt_type_alloc(&allocated_enc_dec_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_ENCRYPT_DECRYPT, action->mech_spec.mechanism, allocated_enc_dec_type);
                enc_dec_type = allocated_enc_dec_type;
            }

            if ((err == SDC_OK) && (action->mech_spec.mechanism_opt_set)) {
                err = sdc_encrypt_decrypt_type_set_opt_bmsk(allocated_enc_dec_type,
                                                            action->mech_spec.mechanism_opt);
            }
        } else {
            enc_dec_type = sdc_encrypt_decrypt_get_default();
        }
    }

    if (err == SDC_OK) {
        if (action->huge_data_opts.enable) {
            err = encrypt_decrypt_init_update_final(action, session, enc_dec_type, &iv_data, log_level);
        } else {
            err = encrypt_decrypt_one_shot(action, session, enc_dec_type, &iv_data, log_level);
        }
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_NOIV) && (iv_data.len > 0)) {
        fprintf(stderr, "User didn't specify IV but iv is returned\n");
        err = SDC_UNKNOWN_ERROR;
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_AUTO_GEN_WRITE_FILE)) {
        /* need to store iv */
        write_file(action->iv_spec.iv_filename, &iv_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_encrypt_decrypt_type_free(allocated_enc_dec_type);

    if (iv_data.data)
        free (iv_data.data);

    return err;
}

static sdc_error_t perform_formatted_encrypt_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_encrypt_decrypt_type_t *allocated_encrypt_decrypt_type = NULL;
    const sdc_encrypt_decrypt_type_t *encrypt_decrypt_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t iv_data;
    sdc_unittest_data_t output_data;
    size_t tag_len;
    action_iv_mode_t iv_mode = ACTION_IV_DEFAULT;

    input_data.data = NULL;
    input_data.len = 0;
    iv_data.data = NULL;
    iv_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, NULL);

    if (err == SDC_OK) {
        iv_mode = action->iv_spec.iv_mode;
        if (iv_mode == ACTION_IV_DEFAULT) {
            iv_mode = ACTION_IV_AUTO_GEN_NO_FILE;
        }

        if (log_level >= LOGGING_VERBOSE) {
            if (iv_mode == ACTION_IV_AUTO_GEN_NO_FILE) {
                printf("Encrypt file %s using key %s, automatically generated iv and write formatted result to %s\n",
                       action->input, key->key_desc,
                       action->output);
            } else {
                if (iv_mode == ACTION_IV_FILE) {
                    printf("Encrypt file %s using key %s, file IV %s and write formatted result to %s\n",
                           action->input, key->key_desc, action->iv_spec.iv_filename,
                           action->output);
                } else {
                    printf("Encrypt file %s using key %s, without IV and write formatted result to %s\n",
                           action->input, key->key_desc,
                           action->output);
                }
            }
        }

        err = read_file(action->input, &input_data, log_level);
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_FILE)) {
        err = read_file_limit(action->iv_spec.iv_filename, &iv_data, action->iv_spec.iv_len, action->iv_spec.iv_len_specified, log_level);
    } else if (iv_mode != ACTION_IV_NOIV) {
        if (action->iv_spec.iv_len_specified) {
            iv_data.len = action->iv_spec.iv_len;
        } else {
            iv_data.len = SDC_IV_USE_DEFAULT;
        }
    }

    if (action->dgst_tag_spec.dgst_tag_len_specified) {
        tag_len = action->dgst_tag_spec.dgst_tag_len;
    } else {
        tag_len = SDC_TAG_USE_DEFAULT;
    }

    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        err = load_session_key_common(session, key, log_level);
    }

    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_encrypt_decrypt_type_alloc(&allocated_encrypt_decrypt_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_ENCRYPT_DECRYPT, action->mech_spec.mechanism, allocated_encrypt_decrypt_type);
                encrypt_decrypt_type = allocated_encrypt_decrypt_type;
            }
        } else {
            encrypt_decrypt_type = sdc_encrypt_decrypt_get_default();
        }
    }

    if (err == SDC_OK) {
        if ((iv_data.len != SDC_IV_USE_DEFAULT) || (tag_len != SDC_TAG_USE_DEFAULT)) {
            err = sdc_encrypt_formatted_extended(session, encrypt_decrypt_type,
                                                 input_data.data, input_data.len,
                                                 iv_data.data, iv_data.len,
                                                 &output_data.data, &output_data.len);
        } else {
            err = sdc_encrypt_formatted(session, encrypt_decrypt_type,
                                        input_data.data, input_data.len,
                                        &output_data.data, &output_data.len);
        }
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_encrypt_decrypt_type_free(allocated_encrypt_decrypt_type);

    if (input_data.data)
        free (input_data.data);
    if (iv_data.data)
        free (iv_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t perform_formatted_decrypt_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    bool format_key;
    sdc_encrypt_decrypt_type_t *encrypt_decrypt_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t output_data;
    sdc_encrypt_decrypt_alg_t alg;
    sdc_encrypt_decrypt_blk_t blk;

    input_data.data = NULL;
    input_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, &format_key);

    if (err == SDC_OK) {
        if (log_level >= LOGGING_VERBOSE) {
            if (format_key) {
                printf("Decrypt file %s using key from format and write result to %s\n",
                       action->input, action->output);
            } else {
                printf("Decrypt file %s using key %s and write result to %s\n",
                       action->input, key->key_desc,
                       action->output);
            }
        }

        err = read_file(action->input, &input_data, log_level);
    }

    if (err == SDC_OK) {
        err = sdc_decrypt_formatted_extract_type(input_data.data, input_data.len, &encrypt_decrypt_type);
    }

    if ((err == SDC_OK) && (log_level >= LOGGING_VERBOSE)) {
        err = sdc_encrypt_decrypt_type_get_algorithm (encrypt_decrypt_type, &alg);

        if (err == SDC_OK) {
            err = sdc_encrypt_decrypt_type_get_block_mode(encrypt_decrypt_type, &blk);

            if (err == SDC_OK) {
                printf("\tFormat algorithm %s, block-mode %s\n",
                       sdc_encrypt_decrypt_algorithm_name(alg),
                       sdc_encrypt_decrypt_block_mode_name(blk));
            }
        }
    }


    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        if (format_key) {
            err = formatted_autoload_key(session,
                                         key, AUTOLOAD_FORMATTED_DECRYPT,
                                         &input_data, log_level);
        } else {
            err = load_session_key_common(session, key, log_level);
        }
    }

    if (err == SDC_OK) {
        err = sdc_decrypt_formatted(session, encrypt_decrypt_type,
                                    input_data.data, input_data.len,
                                    &output_data.data, &output_data.len);
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    if (encrypt_decrypt_type) {
        tmp_err = sdc_encrypt_decrypt_type_free(encrypt_decrypt_type);
        if (err == SDC_OK)
            err = tmp_err;
    }

    if (input_data.data)
        free (input_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t perform_formatted_sign_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_sign_verify_type_t *allocated_sign_verify_type = NULL;
    const sdc_sign_verify_type_t *sign_verify_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t iv_data;
    sdc_unittest_data_t output_data;
    size_t tag_len;
    action_iv_mode_t iv_mode = ACTION_IV_DEFAULT;

    input_data.data = NULL;
    input_data.len = 0;
    iv_data.data = NULL;
    iv_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, NULL);

    if (err == SDC_OK) {
        iv_mode = action->iv_spec.iv_mode;
        if (iv_mode == ACTION_IV_DEFAULT) {
            iv_mode = ACTION_IV_AUTO_GEN_NO_FILE;
        }

        if (log_level >= LOGGING_VERBOSE) {
            if (iv_mode == ACTION_IV_AUTO_GEN_NO_FILE) {
                printf("Sign file %s using key %s, automatically generated iv and write formatted result to %s\n",
                       action->input, key->key_desc,
                       action->output);
            } else {
                if (iv_mode == ACTION_IV_FILE) {
                    printf("Sign file %s using key %s, file IV %s and write formatted result to %s\n",
                           action->input, key->key_desc, action->iv_spec.iv_filename,
                           action->output);
                } else {
                    printf("Sign file %s using key %s, without IV and write formatted result to %s\n",
                           action->input, key->key_desc,
                           action->output);
                }
            }
        }

        err = read_file(action->input, &input_data, log_level);
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_FILE)) {
        err = read_file_limit(action->iv_spec.iv_filename, &iv_data, action->iv_spec.iv_len, action->iv_spec.iv_len_specified, log_level);
    } else if (iv_mode != ACTION_IV_NOIV) {
        if (action->iv_spec.iv_len_specified) {
            iv_data.len = action->iv_spec.iv_len;
        } else {
            iv_data.len = SDC_IV_USE_DEFAULT;
        }
    }

    if (action->dgst_tag_spec.dgst_tag_len_specified) {
        tag_len = action->dgst_tag_spec.dgst_tag_len;
    } else {
        tag_len = SDC_TAG_USE_DEFAULT;
    }

    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        err = load_session_key_common(session, key, log_level);
    }

    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_sign_verify_type_alloc(&allocated_sign_verify_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_SIGN_VERIFY, action->mech_spec.mechanism, allocated_sign_verify_type);
                sign_verify_type = allocated_sign_verify_type;
            }
        } else {
            sign_verify_type = sdc_sign_verify_get_default();
        }
    }

    if (err == SDC_OK) {
        if ((iv_data.len != SDC_IV_USE_DEFAULT) || (tag_len != SDC_TAG_USE_DEFAULT)) {
            err = sdc_sign_formatted_extended(session, sign_verify_type,
                                              input_data.data, input_data.len,
                                              iv_data.data, iv_data.len,
                                              tag_len,
                                              &output_data.data, &output_data.len);
        } else {
            err = sdc_sign_formatted(session, sign_verify_type,
                                     input_data.data, input_data.len,
                                     &output_data.data, &output_data.len);
        }
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_sign_verify_type_free(allocated_sign_verify_type);

    if (input_data.data)
        free (input_data.data);
    if (iv_data.data)
        free (iv_data.data);
    if (output_data.data)
        free (output_data.data);

    return err;
}

static sdc_error_t perform_formatted_verify_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    bool format_key;
    sdc_sign_verify_type_t *sign_verify_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t output_data;
    sdc_sign_verify_alg_t alg;
    sdc_sign_verify_hash_t hash;

    input_data.data = NULL;
    input_data.len = 0;
    output_data.data = NULL;
    output_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, &format_key);

    if (err == SDC_OK) {
        if (log_level >= LOGGING_VERBOSE) {
            if (format_key) {
                printf("Verify file %s using key from format and write result to %s\n",
                       action->input, action->output);
            } else {
                printf("Verify file %s using key %s and write result to %s\n",
                       action->input, key->key_desc,
                       action->output);
            }
        }

        err = read_file(action->input, &input_data, log_level);
    }

    if (err == SDC_OK) {
        err = sdc_verify_formatted_extract_type(input_data.data, input_data.len, &sign_verify_type);
    }

    if ((err == SDC_OK) && (log_level >= LOGGING_VERBOSE)) {
        err = sdc_sign_verify_type_get_alg (sign_verify_type, &alg);

        if (err == SDC_OK) {
            err = sdc_sign_verify_type_get_hash(sign_verify_type, &hash);

            if (err == SDC_OK) {
                printf("\tFormat algorithm %s, block-mode %s\n",
                       sdc_sign_verify_alg_name(alg),
                       sdc_sign_verify_hash_name(hash));
            }
        }
    }


    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        if (format_key) {
            err = formatted_autoload_key(session,
                                         key, AUTOLOAD_FORMATTED_VERIFY,
                                         &input_data, log_level);
        } else {
            err = load_session_key_common(session, key, log_level);
        }
    }

    if (err == SDC_OK) {
        err = sdc_verify_formatted(session, sign_verify_type,
                                   input_data.data, input_data.len,
                                   (const uint8_t **)&output_data.data, &output_data.len);
    }

    if (err == SDC_OK) {
        write_file(action->output, &output_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    if (sign_verify_type) {
        tmp_err = sdc_sign_verify_type_free(sign_verify_type);
        if (err == SDC_OK)
            err = tmp_err;
    }

    if (input_data.data)
        free (input_data.data);

    return err;
}

static sdc_error_t dump_operation_info (char *op_text, const sdc_mechanism_kinds_t kind, logging_t log_level)
{
    sdc_error_t err;
    sdc_mechanism_iterator_t iter;
    sdc_key_fmt_t key_fmt_protect = SDC_KEY_FMT_UNKNOWN;
    sdc_key_fmt_t key_fmt_unprotect = SDC_KEY_FMT_UNKNOWN;
    sdc_key_len_bmsk_t sup_key_lens = SDC_KEY_LEN_BMSK_NONE;
    sdc_key_len_t dflt_key_len = SDC_KEY_LEN_UNKNOWN;

    char keylen_info[KEYLEN_OP_INFO_MAX];
    char mechanism[MECHANISM_NAME_MAX];

    const char *fmt_name_protect = NULL;
    const char *fmt_name_unprotect = NULL;

    (void)log_level;

    printf("%s%s (keylen(bits)):\n",INFO_SEP, op_text);

    err = sdc_helper_mechanism_iterator_init(kind, &iter);

    while (err == SDC_OK) {
        err = sdc_helper_mechanism_get_name(&iter, mechanism, MECHANISM_NAME_MAX);

        if (err == SDC_OK) {
            err = sdc_helper_mechanism_get_key_info(&iter,
                                                    &key_fmt_protect, &key_fmt_unprotect,
                                                    &sup_key_lens, &dflt_key_len);
        }

        if (err == SDC_OK) {
            fmt_name_protect = sdc_key_fmt_get_name(key_fmt_protect);
            fmt_name_unprotect = sdc_key_fmt_get_name(key_fmt_unprotect);

            err = sdc_helper_key_len_str(sup_key_lens, dflt_key_len, keylen_info, KEYLEN_OP_INFO_MAX);
        }

        if (err == SDC_KEYLEN_INVALID) {
            /* key length not relevant for operation */
            printf("\t%s\n", mechanism);
            err = SDC_OK; /* continue */
        } else {
            if (err == SDC_OK)
                printf("\t%20s\t(%s/%s: %s)\n", mechanism, fmt_name_protect, fmt_name_unprotect, keylen_info);
        }

        if (err == SDC_OK) {
            err = sdc_helper_mechanism_iterator_next(&iter);
        }
    }

    if (err == SDC_ALG_MODE_INVALID) {
        /* no more algorithm */
        err = SDC_OK;
    }

    sdc_helper_mechanism_iterator_cleanup(&iter);

    if (err == SDC_OK) {
        printf("%s",INFO_SEP);
    }

    return err;
}

static sdc_error_t sign_verify_get_tag_info(const action_t *action, sdc_unittest_data_t *tag_data, logging_t log_level)
{

    sdc_error_t err = SDC_OK;
    if ((action->operation == ACTION_NON_FORMATTED_VERIFY)) {
        err = read_file_limit(action->dgst_tag_spec.dgst_tag_filename,
                              tag_data, action->dgst_tag_spec.dgst_tag_len,
                              action->dgst_tag_spec.dgst_tag_len_specified,
                              log_level);
    } else {
        if (action->dgst_tag_spec.dgst_tag_len_specified) {
            tag_data->len = action->dgst_tag_spec.dgst_tag_len;
        } else {
            tag_data->len = SDC_TAG_USE_DEFAULT;
        }
    }

    return err;
}

static sdc_error_t sign_verify_one_shot (const action_t *action, sdc_session_t *session,
                                         const sdc_sign_verify_type_t *sign_verify_type,
                                         sdc_unittest_data_t *iv_data, sdc_unittest_data_t *tag_data,
                                         logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_unittest_data_t input_data;

    input_data.data = NULL;
    input_data.len = 0;

    err = read_file(action->input, &input_data, log_level);

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_VERIFY) {
            err = sdc_verify(session, sign_verify_type, input_data.data,
                             input_data.len, iv_data->data, iv_data->len,
                             tag_data->data, tag_data->len);
        } else {
            err = sdc_sign(session, sign_verify_type, input_data.data,
                           input_data.len, &iv_data->data, &iv_data->len,
                           &tag_data->data, &tag_data->len);
        }
    }

    if (input_data.data)
        free(input_data.data);

    return err;
}

static sdc_error_t sign_verify_init_update_final(const action_t *action, sdc_session_t *session,
                                                 const sdc_sign_verify_type_t *sign_verify_type, sdc_unittest_data_t *iv,
                                                 sdc_unittest_data_t *tag_data, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_sign_verify_desc_t *desc = NULL;
    sdc_unittest_data_t in_buf;
    size_t infile_len = 0;
    size_t in_len = 0;
    FILE *in = NULL;

    size_t max_in_len = 0;

    in_buf.data = NULL;
    in_buf.len = 0;

    err = sdc_sign_verify_desc_alloc(&desc);

    if (err == SDC_OK)
        err = sdc_sign_verify_desc_fill(session, sign_verify_type, desc);

    if (err == SDC_OK) {
        if (action->huge_data_opts.chunk_length_set)
            max_in_len = action->huge_data_opts.chunk_length;
        else
            err = sdc_sign_verify_desc_get_max_chunk_len(desc, &(max_in_len));
    }

    if (err == SDC_OK)
        err = open_input(&in, action->input, &infile_len, log_level);

    /* it is useless to allocate more input buffer than the length of the file - so use this as a limit */
    if (max_in_len > infile_len)
        max_in_len = infile_len;

    if (err == SDC_OK)
        err = allocate_input_buffers(&in_buf, max_in_len, log_level);

    if (err == SDC_OK) {
        if (action->operation == ACTION_NON_FORMATTED_VERIFY)
            err = sdc_verify_init(session, sign_verify_type, desc, iv->data,
                                  iv->len);
        else
            err = sdc_sign_init(session, sign_verify_type, desc, &iv->data,
                                &iv->len);
    }

    while (err == SDC_OK) {
        /* fill next input buffer */
        err = read_buffer(in, &in_buf, &in_len, log_level);

        if ((err != SDC_OK) || (in_len == 0))
            break; /* either no more data or error */

        /* process */
        if (action->operation == ACTION_NON_FORMATTED_VERIFY)
            err = sdc_verify_update(session, sign_verify_type, desc, in_buf.data, in_len);
        else
            err = sdc_sign_update(session, sign_verify_type, desc, in_buf.data, in_len);

    }

    if (err == SDC_OK) {
        /* get final output data */
        if (action->operation == ACTION_NON_FORMATTED_VERIFY) {
            err = sdc_verify_finalize(session, sign_verify_type, desc,
                                      tag_data->data, tag_data->len);
        } else {
            err = sdc_sign_finalize(session, sign_verify_type, desc,
                                    &tag_data->data, &tag_data->len);
        }
    }

    if (in)
        fclose(in);
    if (in_buf.data)
        free(in_buf.data);
    if (desc) {
        tmp_err = sdc_sign_verify_desc_free(desc);
        if (err == SDC_OK)
            err = tmp_err;
    }
    return err;
}

static sdc_error_t perform_non_formatted_sign_verify_action (const action_t *action, sdc_ctl_key_t *key, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_sign_verify_type_t *allocated_sign_verify_type = NULL;
    const sdc_sign_verify_type_t *sign_verify_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t iv_data;
    sdc_unittest_data_t tag_data;
    action_iv_mode_t iv_mode = ACTION_IV_DEFAULT;

    input_data.data = NULL;
    input_data.len = 0;
    iv_data.data = NULL;
    iv_data.len = 0;
    tag_data.data = NULL;
    tag_data.len = 0;

    err = check_key_arguments_common(&action->key_spec, key, NULL);

    if (err == SDC_OK) {
        iv_mode = action->iv_spec.iv_mode;
        if (iv_mode == ACTION_IV_DEFAULT) {
            iv_mode = ACTION_IV_NOIV;
        }

        if (log_level >= LOGGING_VERBOSE) {
            if (action->operation == ACTION_NON_FORMATTED_SIGN) {
                if (iv_mode == ACTION_IV_AUTO_GEN_WRITE_FILE) {
                    printf("Sign file %s using key %s, automatically generated iv and write iv to %s and tag to %s\n",
                           action->input, key->key_desc,
                           action->iv_spec.iv_filename, action->dgst_tag_spec.dgst_tag_filename);
                } else {
                    if (iv_mode == ACTION_IV_FILE) {
                        printf("Sign file %s using key %s, file IV %s and write tag to %s\n",
                               action->input, key->key_desc, action->iv_spec.iv_filename,
                               action->dgst_tag_spec.dgst_tag_filename);
                    } else {
                        printf("Sign file %s using key %s, without IV and write tag to %s\n",
                               action->input, key->key_desc,
                               action->dgst_tag_spec.dgst_tag_filename);
                    }
                }
            } else {
                printf("Verify file %s using key %s, file IV %s, tag %s\n",
                       action->input, key->key_desc, action->iv_spec.iv_filename, action->dgst_tag_spec.dgst_tag_filename);
            }
        }
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_FILE)) {
        err = read_file_limit(action->iv_spec.iv_filename, &iv_data, action->iv_spec.iv_len, action->iv_spec.iv_len_specified, log_level);
    } else if (iv_mode != ACTION_IV_NOIV) {
        if (action->iv_spec.iv_len_specified) {
            iv_data.len = action->iv_spec.iv_len;
        } else {
            iv_data.len = SDC_IV_USE_DEFAULT;
        }
    }

    if (err == SDC_OK) {
        err = sdc_open_session(&session);
    }

    if (err == SDC_OK) {
        err = load_session_key_common(session, key, log_level);
    }

    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_sign_verify_type_alloc(&allocated_sign_verify_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_SIGN_VERIFY, action->mech_spec.mechanism, allocated_sign_verify_type);
                sign_verify_type = allocated_sign_verify_type;
            }

            if ((err == SDC_OK) && (action->mech_spec.mechanism_opt_set)) {
                err = sdc_sign_verify_type_set_opt_bmsk(allocated_sign_verify_type,
                                                        action->mech_spec.mechanism_opt);
            }
        } else {
            sign_verify_type = sdc_sign_verify_get_default();
        }
    }

    if (err == SDC_OK)
        err = sign_verify_get_tag_info(action, &tag_data, log_level);

    if (err == SDC_OK) {
        if (action->huge_data_opts.enable) {
            err = sign_verify_init_update_final(action, session, sign_verify_type,
                                                &iv_data, &tag_data, log_level);
        } else {
            err = sign_verify_one_shot(action, session, sign_verify_type,
                                       &iv_data, &tag_data, log_level);
        }
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_NOIV) && (iv_data.len > 0)) {
        fprintf(stderr, "User didn't specify IV but iv is returned\n");
        err = SDC_UNKNOWN_ERROR;
    }

    if ((err == SDC_OK) && (iv_mode == ACTION_IV_AUTO_GEN_WRITE_FILE)) {
        /* need to store iv */
        write_file(action->iv_spec.iv_filename, &iv_data, log_level);
    }

    if ((err == SDC_OK) && (action->operation == ACTION_NON_FORMATTED_SIGN)) {
        /* need to store tag */
        write_file(action->dgst_tag_spec.dgst_tag_filename, &tag_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_sign_verify_type_free(allocated_sign_verify_type);

    if (input_data.data)
        free (input_data.data);
    if (iv_data.data)
        free (iv_data.data);
    if (tag_data.data)
        free (tag_data.data);

    return err;
}

static void get_digest_len_info(const action_t *action, sdc_unittest_data_t *dgst_data)
{
     if (action->dgst_tag_spec.dgst_tag_len_specified) {
         dgst_data->len = action->dgst_tag_spec.dgst_tag_len;
     } else {
         dgst_data->len = SDC_TAG_USE_DEFAULT;
     }
}

static sdc_error_t dgst_one_shot (const action_t *action,
                                  sdc_session_t *session,
                                  const sdc_dgst_type_t *dgst_type,
                                  sdc_unittest_data_t *dgst_data,
                                  logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_unittest_data_t input_data;

    input_data.data = NULL;
    input_data.len = 0;

    err = read_file(action->input, &input_data, log_level);
    if ((err == SDC_OK) && (action->operation == ACTION_NON_FORMATTED_DGST)) {
        err = sdc_dgst(session, dgst_type,
                       input_data.data, input_data.len,
                       &dgst_data->data, &dgst_data->len);
    }

    if (input_data.data)
        free (input_data.data);

    return err;
}

static sdc_error_t dgst_init_update_final(const action_t *action, sdc_session_t *session,
                                          const sdc_dgst_type_t *dgst_type, sdc_unittest_data_t *dgst_data, logging_t log_level)
 {
     sdc_error_t err = SDC_OK;
     sdc_error_t tmp_err;
     sdc_dgst_desc_t *desc = NULL;
     sdc_unittest_data_t in_buf;
     size_t infile_len = 0;
     size_t in_len = 0;
     size_t max_in_len = 0;
     FILE *in = NULL;

     in_buf.data = NULL;
     in_buf.len = 0;

     err = sdc_dgst_desc_alloc(&desc);
     if (err == SDC_OK)
         err = sdc_dgst_desc_fill(dgst_type, desc);

     if (err == SDC_OK) {
         if (action->huge_data_opts.chunk_length_set)
             max_in_len = action->huge_data_opts.chunk_length;
         else
             err = sdc_dgst_desc_get_max_chunk_len(desc, &(max_in_len));
     }

     if (err == SDC_OK)
         err = open_input(&in, action->input, &infile_len, log_level);

     /* it is useless to allocate more input buffer than the length of the file - so use this as a limit */
     if (max_in_len > infile_len)
         max_in_len = infile_len;

     if (err == SDC_OK)
         err = allocate_input_buffers(&in_buf, max_in_len, log_level);

     if ((err == SDC_OK) && (action->operation == ACTION_NON_FORMATTED_DGST)) {
         err = sdc_dgst_init(session, dgst_type, desc);
     }

     while (err == SDC_OK) {
         /* fill next input buffer */
         err = read_buffer(in, &in_buf, &in_len, log_level);

         if ((err != SDC_OK) || (in_len == 0))
             break; /* either no more data or error */

         /* process */
         if (action->operation == ACTION_NON_FORMATTED_DGST)
             err = sdc_dgst_update(session, dgst_type, desc, in_buf.data, in_len);
     }

     if ((err == SDC_OK) && (action->operation == ACTION_NON_FORMATTED_DGST)) {
        /* get final digest data */
        err = sdc_dgst_finalize(session, dgst_type,
                                desc, &dgst_data->data, &dgst_data->len);
     }

     if (in)
         fclose(in);
     if (in_buf.data)
         free(in_buf.data);
     if (desc) {
         tmp_err = sdc_dgst_desc_free(desc);
         if (err == SDC_OK)
             err = tmp_err;
     }
     return err;
 }

static sdc_error_t perform_non_formatted_dgst_action (const action_t *action, logging_t log_level)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;
    sdc_dgst_type_t *allocated_dgst_type = NULL;
    const sdc_dgst_type_t *dgst_type = NULL;
    sdc_unittest_data_t input_data;
    sdc_unittest_data_t dgst_data;

    input_data.data = NULL;
    input_data.len = 0;
    dgst_data.data = NULL;
    dgst_data.len = 0;

    if (log_level >= LOGGING_VERBOSE) {
        printf("Digest of file %s, digest to %s\n",
               action->input, action->dgst_tag_spec.dgst_tag_filename);
    }

    err = sdc_open_session(&session);
    if (err == SDC_OK) {
        if (action->mech_spec.mechanism) {
            err = sdc_dgst_type_alloc(&allocated_dgst_type);

            if (err == SDC_OK) {
                err = sdc_helper_set_mechanism_by_name(SDC_DGST, action->mech_spec.mechanism, allocated_dgst_type);
                dgst_type = allocated_dgst_type;
            }

            if ((err == SDC_OK) && (action->mech_spec.mechanism_opt_set)) {
                err = sdc_dgst_type_set_opt_bmsk(allocated_dgst_type,
                                                 action->mech_spec.mechanism_opt);
            }
        } else {
            dgst_type = sdc_dgst_get_default();
        }
    }

    get_digest_len_info(action, &dgst_data);

    if (err == SDC_OK) {
        if (action->huge_data_opts.enable) {
            err = dgst_init_update_final(action, session, dgst_type, &dgst_data, log_level);
        } else {
            err = dgst_one_shot(action, session, dgst_type, &dgst_data, log_level);
        }
    }
    if (err == SDC_OK) {
        /* store digest */
        write_file(action->dgst_tag_spec.dgst_tag_filename, &dgst_data, log_level);
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    sdc_dgst_type_free(allocated_dgst_type);

    if (input_data.data)
        free (input_data.data);
    if (dgst_data.data)
        free (dgst_data.data);

    return err;
}

static sdc_error_t perform_rng_ops (const action_t *action, logging_t log_level)
{
    const action_option_rng_t *opt = &action->options.rng_options;
    sdc_error_t err = SDC_OK;
    sdc_error_t tmp_err;
    sdc_session_t *session = NULL;

    sdc_unittest_data_t gen_data;

    gen_data.data = NULL;
    gen_data.len = opt->gen_len;

    err = sdc_open_session(&session);

    if (opt->gen) {
        if (err == SDC_OK) {
            err = sdc_random_gen_buffer(session, gen_data.len, &(gen_data.data));
        }

        if (err == SDC_OK) {
            err = write_file(action->output, &gen_data, log_level);
        }
    }

    tmp_err = sdc_close_session(session);
    if (err == SDC_OK)
        err = tmp_err;

    if (gen_data.data)
        free (gen_data.data);

    return err;
}

static sdc_error_t dump_arch_key_info_for_format(sdc_key_fmt_t fmt)
{
    sdc_key_len_bmsk_t keylens;
    sdc_key_len_t keylen_val;
    size_t keylen_bits;
    size_t keylen_bytes = 0;
    sdc_error_t err = SDC_OK;
    const char *fmt_name;

    err = sdc_key_len_bmsks(fmt, &keylens);
    if ((err != SDC_OK) || (keylens == SDC_KEY_LEN_BMSK_NONE))
        return err;

    fmt_name = sdc_key_fmt_get_name(fmt);
    printf("\t%s:\n",fmt_name);

    err = sdc_key_len_bmsk_first(keylens, &keylen_val);

    while ((err == SDC_OK) && (keylen_val != SDC_KEY_LEN_UNKNOWN)) {

        err = sdc_key_len_get_bits(keylen_val, &keylen_bits);

        if (err == SDC_OK) {
            err = sdc_key_len_get_bytes(keylen_val, &keylen_bytes);
        }

        if (err == SDC_OK) {
            printf("\t\t%zubits = %zubytes\n", keylen_bits, keylen_bytes);

            err = sdc_key_len_bmsk_next(keylens, &keylen_val);
        }
    }

    printf("\n");

    return err;
}

static sdc_error_t dump_arch_key_info()
{
    sdc_key_fmt_t fmt;
    sdc_error_t err = SDC_OK;

    printf("%sSupported key length:\n\n",INFO_SEP);

    err = sdc_key_fmt_bmsk_first(SDC_KEY_FMT_BMSK_ALL, &fmt);

    while ((err == SDC_OK) && (fmt != SDC_KEY_FMT_UNKNOWN)) {
        err = dump_arch_key_info_for_format(fmt);

        if (err == SDC_OK)
            err = sdc_key_fmt_bmsk_next(SDC_KEY_FMT_BMSK_ALL, &fmt);
    }

    printf("%s",INFO_SEP);

    return err;
}

static sdc_error_t dump_arch_errors (logging_t log_level)
{
    const char *err_str;
    sdc_error_t err;

    (void)log_level;

    printf("List error codes for architecture:");
    err = SDC_ERROR_FIRST;
    while (err <= SDC_ERROR_LAST) {
        err_str = sdc_get_error_string(err);

        if (err_str != NULL) {
            printf("\t%u : %s\n", err, err_str);
        }

        err++;
    }

    return SDC_OK;
}

static sdc_error_t dump_arch_error (sdc_error_t err, logging_t log_level)
{
    const char *err_str;

    (void)log_level;

    err_str = sdc_get_error_string(err);

    printf("Resolve error code:");

    printf("\t%u : %s\n", err, err_str);

    return SDC_OK;
}

static sdc_error_t perform_arch_info_action (const action_t *action, logging_t log_level)
{
    const action_option_arch_info_t *opt = &action->options.arch_info_options;
    sdc_error_t err = SDC_OK;

    if (opt->dump_arch_key_lengths) {
        err = dump_arch_key_info();
    }

    if ((err == SDC_OK) && (opt->dump_wrap_alg_blk)) {
        err = dump_operation_info("Supported wrap/unwrap algorithms/block-modes", SDC_WRAP_UNWRAP, log_level);
    }
    if ((err == SDC_OK) && (opt->dump_encrypt_alg_blk)) {
        err = dump_operation_info("Supported encrypt/decrypt algorithms/block-modes", SDC_ENCRYPT_DECRYPT, log_level);
    }
    if ((err == SDC_OK) && (opt->dump_signverify_alg_hash)) {
        err = dump_operation_info("Supported sign/verify algorithm/hash-modes", SDC_SIGN_VERIFY, log_level);
    }
    if ((err == SDC_OK) && (opt->dump_digest_hash)) {
        err = dump_operation_info("Supported digest hash-modes", SDC_DGST, log_level);
    }
    if ((err == SDC_OK) && (opt->dump_error_list)) {
        err = dump_arch_errors(log_level);
    }
    if ((err == SDC_OK) && (opt->dump_error_code)) {
        err = dump_arch_error(opt->error_code, log_level);
    }

    return err;
}


int main(int argc, char **argv)
{
    arguments_t arguments;
    action_t *curr_action, *next_action;
    size_t action_cnt = 0;
    sdc_error_t sdc_err = SDC_OK;
    sdc_ctl_key_t last_key;

    key_invalidate(&last_key);

    /* Parse the command line args */
    sdc_err = sdc_ctl_parse_args(argc, argv, &arguments);

    curr_action = arguments.first_action;

    while ((curr_action != NULL) && (sdc_err == SDC_OK)) {
        if (arguments.log_level >= LOGGING_VERBOSE) {
            printf ("\nPerform action %zu:\n", ++action_cnt);
        }

        switch (curr_action->operation) {
        case ACTION_INSTALL:
            sdc_err = perform_install_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_REMOVE:
            sdc_err = perform_remove_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_NON_FORMATTED_UNWRAP:
        case ACTION_NON_FORMATTED_WRAP:
            sdc_err = perform_non_formatted_wrap_unwrap_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_FORMATTED_WRAP:
            sdc_err = perform_formatted_wrap_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_FORMATTED_UNWRAP:
            sdc_err = perform_formatted_unwrap_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_NON_FORMATTED_ENCRYPT:
        case ACTION_NON_FORMATTED_DECRYPT:
            sdc_err = perform_non_formatted_encrypt_decrypt_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_FORMATTED_ENCRYPT:
            sdc_err = perform_formatted_encrypt_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_FORMATTED_DECRYPT:
            sdc_err = perform_formatted_decrypt_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_NON_FORMATTED_SIGN:
        case ACTION_NON_FORMATTED_VERIFY:
            sdc_err = perform_non_formatted_sign_verify_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_FORMATTED_SIGN:
            sdc_err = perform_formatted_sign_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_FORMATTED_VERIFY:
            sdc_err = perform_formatted_verify_action (curr_action, &last_key, arguments.log_level);
            break;
        case ACTION_NON_FORMATTED_DGST:
            sdc_err = perform_non_formatted_dgst_action(curr_action, arguments.log_level);
            break;
        case ACTION_RNG:
            sdc_err = perform_rng_ops (curr_action, arguments.log_level);
            break;
        case ACTION_ARCH_INFO:
            sdc_err = perform_arch_info_action (curr_action, arguments.log_level);
            break;
        default:
            if (arguments.log_level > LOGGING_QUIET)
                fprintf(stderr, "Unknown operation\n");
            sdc_err = SDC_UNKNOWN_ERROR;
        }

        next_action = curr_action->next;
        free(curr_action);
        curr_action = next_action;
    }

    if (sdc_err == SDC_OK) {
        return 0;
    } else {
        if (arguments.log_level > LOGGING_QUIET)
            fprintf(stderr, "Error %i : %s\n", sdc_err, sdc_get_error_string(sdc_err));
        return sdc_err;
    }
}

